/*  Berechnet für einen gegebenen AG die notwendige effektive Arbeitszeitzeit und beachtet dabei, ob es sich um eine
    Auswärtskostenstelle handelt, gestempelte Zeiten und manuelle Korrekturen der notwendige Arbeitszeit.

    Im Unterschied zur Funktion scheduling.resource_timeline__ab2__get_scheduled_time() berechnet diese Funktion die notwendige
    Arbeitszeit aus der Spalte ab2.a2_ta bzw. ab2_wkstplan.a2w_stukorr und beachtet damit, ob die notwendige Arbeitszeit manuell
    überschrieben wurde.
    Wenn der Parameter _include_stemp_time auf true steht, werden die Stempelzeiten beachtet und damit die _noch_ notwendige
    effektive Arbeitszeitzeit ermittelt.
*/
SELECT tsystem.function__drop_by_regex( 'ab2__required_worktime__get', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.ab2__required_worktime__get(
      _ab2_record ab2,                                    -- Der Arbeitsgang (ID) für den die noch notwendige Arbeitszeit ermittelt werden soll.
      _include_stemp_time bool = true,                    -- Sollen die Stempeltzeiten (schon erledigte Abarbeitungszeit) beachtet werden?
      _include_stemp_time_before timestamp = 'infinity'   -- Der maximale Endzeitpunkt von Stempeleinträgen bis zu dem Stempelungen berücksichtigt werden.
  ) RETURNS numeric AS $$
  DECLARE
      _ab2_wkstplan record;

      _a2_ta_bedarfsag CONSTANT numeric := 4.0;           -- Standardlaufzeit für Bedarfs-AG. Diese wird in der Terminerung für Bedarfs-AG verwendet.

      _time_required        numeric = 0; -- Notwendige Abarbeitungszeit.
      _time_still_required  numeric = 0; -- Noch notwendige Arbeitszeit unter Beachtung von erfolgten Stempelzeiten (schon erledigter Abarbeitungszeit).
  BEGIN

      IF _ab2_record.a2_id IS null THEN
          RETURN null;
      END IF;
      -- AG beendet -> keine benötigte Zeit
      IF ( _ab2_record.a2_ende ) THEN
          RETURN 0;
      END IF;

      -- Prüfung, ob es eine Auswärtskostenstelle ist.
      IF  _ab2_record.a2_ausw THEN

        -- dlz specified in days
        -- if non is defined on the ab2 than use ksv.ks_dlz
        -- it shouldn't be possible, that both of those values are undefined. Is prevented by a trigger.
        -- if already send, the time from first send to now is alreday done. at least one day? or zero?
        -- TODO ci-test (in past with ausw)
        RETURN  greatest( (   _ab2_record.a2_dlz
                           - CASE WHEN _include_stemp_time THEN
                                 (current_date - coalesce(tplanterm.ausw_menge_send_dat(_ab2_record.a2_id), current_date) )::integer -- keine Rückgabe = null => wir wollen keine Differenz
                             ELSE
                                 0
                             END
                            )
                          , 0.25
                         ) * 24 * 60 * 60 ;
      END IF;

      SELECT a2w_stukorr, a2w_resource_ks_abt_main_fix INTO _ab2_wkstplan
        FROM ab2_wkstplan
       WHERE a2w_a2_id = _ab2_record.a2_id AND NOT a2w_marked = -1; -- SplitAGs;

      -- this is in hours
      IF _ab2_record.a2_ta > 0 THEN
            _time_required := _ab2_record.a2_ta * 60 * 60;
      ELSE -- _ab2_record.a2_ta = 0
        IF ks_plan FROM ksv WHERE ks_abt = coalesce(_ab2_wkstplan.a2w_resource_ks_abt_main_fix, _ab2_record.a2_ks)
        THEN
            _time_required := _a2_ta_bedarfsag * 60 * 60; -- Standardlaufzeit für Bedarfs-AGe ansetzen, dabei von Info-AGen unterscheiden
        ELSE
            _time_required := 0;
        END IF;
      END IF;

      -- apply overriden time requirements from a2w
      IF ( _ab2_wkstplan.a2w_stukorr IS NOT NULL ) THEN
          _time_required := _ab2_wkstplan.a2w_stukorr * 60 * 60;
      END IF;

      -- quit if no a2_stemp is not needed
      IF ( NOT _include_stemp_time ) THEN
          RETURN _time_required;
      END IF;

      -- Die noch benötigte Arbeitszeit des AG ergibt sich aus Differenz der geplanten und der schon erledigten Arbeitszeit.
      -- [x] AXS: a2_ta bzw. a2w_stukorr sind effektive Zeitangaben und ab2__worktime_done__get() ist nicht effektive Zeiten. Bei der Brechnung werden diese vermischt. Fehler?
      --> Berichtigt. ab2__worktime_done__get() geändert. Gibt nun effektive Werte zurück.
      IF _include_stemp_time_before  = 'infinity' THEN

         _time_still_required :=
            _time_required - _ab2_record.a2_time_stemp * ( 60 * 60 );

      ELSE

        _time_still_required :=
            _time_required -
            scheduling.ab2__worktime_done__get( _ab2_record, _include_stemp_time_before );

      END IF;

      -- Die erledigte Arbeitszeit des AG ist größer als die geplante Arbeitszeit.
      IF ( _time_still_required <= 0 ) THEN
        -- [x] AXS: Sind diese 4 Stunden aus der Transportmatrix?
        --> Nein. Als Standard für die noch übrige Arbeitszeit wird eine halbe Schicht (=4h) angenommen, wenn der AG noch nicht
        --> als beendet (ab2.a2_ende) markiert, aber die komplette Zeit gestempelt wurde. In diesen 4h sollte eine Reaktion
        --> erfolgen (z.B AG. als beendet markieren oder neue notwendige Zeit eintragen).
        -- ACHTUNG: Das bedeutet, dass beim Überschreiten der geplanten Arbeitszeit die noch notwendige Arbeitszeit von wenigen Sekunden zu 4 Stunden (halbe Schichtzeit) springt!
        -- Wird wahrscheinlich in der Oberfläche auch so angezeigt. Daniel weigert sich aktuell darüber gedanken zu machen.
        -- TODO AXS: Dieses Verhalten in der Oberfläche ansehen, dann übedenken und ggf. den Code an eine andere Stelle verschieben. Wir wollen den Ablauf der geplanten Arbeitszeit in der Oberfläche sehen!
        -- Wenn es keine Vorgabezeiten gibt (Info-Arbeitsgänge) dann bleiben wir bei 0!
        IF _ab2_record.a2_ta > 0 THEN
           _time_still_required := least(4, _ab2_record.a2_ta / 2) * 60 * 60; -- wir nehmen 4 Stunden an, oder wenn die Vorgabe kleiner als 8 Stunden ist - dann halbe Vorgabezeit! Langsame Annäherung an 4 Stunden (halbe Schicht) bei voll gestempelten aber nicht beendeten AGs unter 8 Stunden.
        ELSE
           _time_still_required := 0;
        END IF;
      END IF;

      -- worktimes specified in hours
      RETURN ceil(_time_still_required);

  END $$ LANGUAGE plpgsql STABLE;


  CREATE OR REPLACE FUNCTION scheduling.ab2__required_worktime__get(
      _ab2_id integer,
      _include_stemp_time bool = true,
      _include_stemp_time_before timestamp = 'infinity'
      )
      RETURNS numeric
      AS $$
          -- _ab2_record befüllen.
          SELECT scheduling.ab2__required_worktime__get(
                 ab2,
                 _include_stemp_time,
                 _include_stemp_time_before
                 )
            FROM ab2
           WHERE a2_id = _ab2_id;
      $$ LANGUAGE sql STABLE;